/*"Gear Indicator": Simple OutGauge client for LFS
 * Author: Vladimir Kadlec
 *         vladimirkadlec@gmail.com
 *         http://www.fi.muni.cz/~xkadlec
 * Updated by:  Nick Aronson 29/05/07
 *              nick (dot) a (dot) aronson (at) gmail (dot) com
 * New feature: 5 LED rev meter outputs added
 */
/*
This file is part of Gear Indicator.

Gear Indicator is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

Gear Indicator is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Gear Indicator; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "log.h"
#include "config.h"

#define MAX_IP_SIZE 100
unsigned char SEG_N[NUMBER_OF_GEARS] =
         {129, 131, 192, 87, 213, 225, 181, 183, 196, 247};

void init_config (Config *conf) {
int i;
    conf->ip = calloc (MAX_IP_SIZE, 1);
    strcpy (conf->ip, "127.0.0.1");
    conf->port = 11111;
    conf->lpt_base = 0x378;
    memcpy (conf->display, SEG_N, NUMBER_OF_GEARS);
    conf->shift_light = 8;
    conf->debug = 1;
    conf->disp_off = 0;
    conf->blink_time = 200;
    conf->fuel_warning = 5;
    conf->ext_off = 0;
    for (i = 0; i < 5; i++) {  // sizeof (PINS)
        conf->pin_func[i] = F_NONE;
        conf->pin_func_neg[i] = 0;
        conf->pin_lfs_flag[i] = 0;
    }
}

char *func_to_str (LFS_FUNCS f) {
    switch (f) {
        case F_LOW_FUEL:
            return ("low_fuel");
        break;
        case F_SHIFT_LIGHT:
            return ("shift_light");
        break;
        case F_TRACTION_CONTROL:
            return ("traction_control");
        break;
        case F_SIGNAL_LIGHTS:
            return ("signal_lights");
        break;
        case F_HANDBRAKE:
            return ("handbrake");
        break;
        case F_RPM_LIMITER:
            return ("rpm_limiter");
        break;
        case F_REDLINE:
            return ("redline");
        break;
        case F_NONE:
            return ("none");
        break;
        case F_RPM1:
            return ("rpm1");
        break;
        case F_RPM2:
            return ("rpm2");
        break;
        case F_RPM3:
            return ("rpm3");
        break;
        case F_RPM4:
            return ("rpm4");
        break;
        case F_RPM5:
            return ("rpm5");
        break;
        default:
            return ("unknown");
    }
}

void print_config (Config *conf) {
int i;
int p[5] = {1, 9, 14, 16, 17};  //sizeof (PINS)

    printf ("Current configuration:\n");
    printf ("IP:%s Port:%d LPT:0x%x\n", conf->ip, conf->port,
            conf->lpt_base);
    printf ("Display:");
    for (i = 0; i < NUMBER_OF_GEARS; i++) {
        printf (" %d", conf->display[i]);
    }
    printf ("\nShift Light:%d, Blink time:%d, Fuel warning:%d%%\n",
        conf->shift_light, conf->blink_time, conf->fuel_warning);
    printf ("Debug info:%d    Display off code:%d\n",
        conf->debug, conf->disp_off);
    printf ("Pin assignments:\n");
    for (i = 0; i < 5 ; i++) {  //  sizeof (PINS)
        printf ("PIN_%d: %s neg=%d lfs=%d\n", p[i], func_to_str (conf->pin_func[i]),
                    conf->pin_func_neg[i], conf->pin_lfs_flag[i]);
    }
    printf ("Extended pins off code:%d\n", conf->ext_off);
}

void read_line (char *file_buf, long *file_pos, char **line_buf)
{
    long i;
    *line_buf = file_buf + (*file_pos);
    i = *file_pos;                                                                  while ((file_buf[i] != '\n') && (file_buf[i] != 0))
        i++;
    file_buf[i] = 0;
    *file_pos = i + 1;
}

LFS_FUNCS str_to_func (char *wrd, int line_no, word *lfs_func) {
    if (strcasecmp (wrd, "low_fuel") == 0) {
        *lfs_func = 0;
        return (F_LOW_FUEL);
    }
    if (strcasecmp (wrd, "shift_light") == 0) {
        *lfs_func = OG_SHIFTLIGHT;
        return (F_SHIFT_LIGHT);
    }
    if (strcasecmp (wrd, "traction_control") == 0) {
        *lfs_func = OG_TC;
        return (F_TRACTION_CONTROL);
    }
    if (strcasecmp (wrd, "signal_lights") == 0) {
        *lfs_func = OG_SIGNAL_L | OG_SIGNAL_R;
        return (F_SIGNAL_LIGHTS);
    }
    if (strcasecmp (wrd, "handbrake") == 0) {
        *lfs_func = OG_HANDBRAKE;
        return (F_HANDBRAKE);
    }
    if (strcasecmp (wrd, "rpm_limiter") == 0) {
        *lfs_func = OG_PITSPEED;
        return (F_RPM_LIMITER);
    }
    if (strcasecmp (wrd, "redline") == 0) {
        *lfs_func = OG_REDLINE;
        return (F_REDLINE);
    }
    if (strcasecmp (wrd, "rpm1") == 0) {
        *lfs_func = 0;
        return (F_RPM1);
    }
    if (strcasecmp (wrd, "rpm2") == 0) {
        *lfs_func = 0;
        return (F_RPM2);
    }
    if (strcasecmp (wrd, "rpm3") == 0) {
        *lfs_func = 0;
        return (F_RPM3);
    }
    if (strcasecmp (wrd, "rpm4") == 0) {
        *lfs_func = 0;
        return (F_RPM4);
    }
    if (strcasecmp (wrd, "rpm5") == 0) {
        *lfs_func = 0;
        return (F_RPM5);
    }
    if (strcasecmp (wrd, "none") == 0) {
        *lfs_func = 0;
        return (F_NONE);
    }
    *lfs_func = 0;
    warning ("read_config, line: %d, Wrong name of a function\n",
                             line_no);
    return (F_NONE);
}

typedef enum {S_NONE, S_IP, S_PORT, S_LPT, S_SEG_N, S_SEG_N_N, S_SEG_DOT,
S_DEBUG, S_SEG_OFF, S_BLINK, S_FUEL, S_PIN_1, S_PIN_14, S_PIN_16, S_PIN_17, S_PIN_9,
S_EXT_OFF}
STATE;

int read_config (char *file_name, Config *conf) {
FILE *f;
char *file_buf, *line_buf, *wrd;
long file_size, file_pos = 0, line_no = 0;
STATE state = S_NONE;
int port, segment_no = -1;
char *err_port;
LFS_FUNCS function;
int in_pin = 0;
word lfs_flags;

    if ((f = fopen (file_name, "r")) == NULL) {
        error ("Can't open file: %s.\n", file_name);
        return (0);
    }
    fseek (f, 0, SEEK_END);
    file_size = ftell (f);
    fseek (f, 0, SEEK_SET);
    if ((file_buf = (char *) calloc (file_size+1, 1)) == NULL) {
        error ("Can't allocate memory for config file.\n");
        return (0);
    }
    file_size = fread (file_buf, 1, file_size, f);
    fclose (f);

    while (file_pos < file_size) {
        if (state != S_NONE) {
            if (state != S_SEG_N_N) {
                error ("read_config, invalid state, please report.\n");
                free (file_buf);
                return (0);
            }
            warning ("read_config, line: %d,"
                           " too litle numbers for display\n", line_no);
        }
        read_line (file_buf, &file_pos, &line_buf);
        line_no++;
        state = S_NONE;
        wrd = strtok (line_buf, " \t\n\r");
        for (; 1; wrd = strtok (NULL, " \t\n\r")) {
            if (wrd == NULL)
                break;
            if (wrd[0] == 0)
                continue;
            if (wrd[0] == '#')
                break;
            switch (state) {
            case S_DEBUG:
               port = strtol (wrd, &err_port, 10);
                if (((*err_port) != 0) || ((port != 0) && (port != 1))) {
                    warning ("read_config, line: %d,"
                              " Invalid debug option (should be 1 or 0)\n",
                              line_no);
                    state = S_NONE;
                    continue;
                }
                conf->debug = port;
                state = S_NONE;
            break;
            case S_SEG_OFF:
               port = strtol (wrd, &err_port, 10);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid display number\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->disp_off = port;
                state = S_NONE;
            break;
            case S_BLINK:
               port = strtol (wrd, &err_port, 10);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid blink time\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->blink_time = port;
                state = S_NONE;
            break;
            case S_IP:
               strncpy (conf->ip, wrd, MAX_IP_SIZE);
               state = S_NONE;
            break;
            case S_PORT:
                port = strtol (wrd, &err_port, 10);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid port number\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->port = port;
                state = S_NONE;
            break;
            case S_LPT:
                port = strtol (wrd, &err_port, 16);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid LPT port number\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->lpt_base = port;
                state = S_NONE;
            break;
            case S_SEG_N:
                segment_no = 0;
                port = strtol (wrd, &err_port, 10);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid display number\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->display[segment_no] = port;
                segment_no++;
                state = S_SEG_N_N;
            break;
            case S_SEG_N_N:
                port = strtol (wrd, &err_port, 10);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid display number\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->display[segment_no] = port;
                segment_no++;
                if (segment_no == NUMBER_OF_GEARS) state = S_NONE;
            break;
            case S_SEG_DOT:
                port = strtol (wrd, &err_port, 10);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid display number\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->shift_light = port;
                state = S_NONE;
            break;
            case S_FUEL:
                port = strtol (wrd, &err_port, 10);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid fuel warning number\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->fuel_warning = port;
                state = S_NONE;
            break;
            /*ext pins*/
            case S_PIN_1:
                if (!in_pin) {
                    in_pin = 1;
                    function = str_to_func (wrd, line_no, &lfs_flags);
                    conf->pin_func[PIN_1] = function;
                    conf->pin_lfs_flag[PIN_1] = lfs_flags;
                    continue;
                }
                in_pin = 0;
                port = strtol (wrd, &err_port, 10);
                if (((*err_port) != 0) || ((port != 0) && (port != 1))) {
                    warning ("read_config, line: %d, Invalid pin neg_value"
                            " (should be 1 or 0)\n", line_no);
                    state = S_NONE;
                    continue;
                }
                conf->pin_func_neg[PIN_1] = port;
                state = S_NONE;
            break;
            case S_PIN_9:
                if (!in_pin) {
                    in_pin = 1;
                    function = str_to_func (wrd, line_no, &lfs_flags);
                    conf->pin_func[PIN_9] = function;
                    conf->pin_lfs_flag[PIN_9] = lfs_flags;
                    continue;
                }
                in_pin = 0;
                port = strtol (wrd, &err_port, 10);
                if (((*err_port) != 0) || ((port != 0) && (port != 1))) {
                    warning ("read_config, line: %d, Invalid pin neg_value"
                            " (should be 1 or 0)\n", line_no);
                    state = S_NONE;
                    continue;
                }
                conf->pin_func_neg[PIN_9] = port;
                state = S_NONE;
            break;
            case S_PIN_14:
                if (!in_pin) {
                    in_pin = 1;
                    function = str_to_func (wrd, line_no, &lfs_flags);
                    conf->pin_func[PIN_14] = function;
                    conf->pin_lfs_flag[PIN_14] = lfs_flags;
                    continue;
                }
                in_pin = 0;
                port = strtol (wrd, &err_port, 10);
                if (((*err_port) != 0) || ((port != 0) && (port != 1))) {
                    warning ("read_config, line: %d, Invalid pin neg_value"
                            " (should be 1 or 0)\n", line_no);
                    state = S_NONE;
                    continue;
                }
                conf->pin_func_neg[PIN_14] = port;
                state = S_NONE;
            break;
            case S_PIN_16:
                if (!in_pin) {
                    in_pin = 1;
                    function = str_to_func (wrd, line_no, &lfs_flags);
                    conf->pin_func[PIN_16] = function;
                    conf->pin_lfs_flag[PIN_16] = lfs_flags;
                    continue;
                }
                in_pin = 0;
                port = strtol (wrd, &err_port, 10);
                if (((*err_port) != 0) || ((port != 0) && (port != 1))) {
                    warning ("read_config, line: %d, Invalid pin neg_value"
                            " (should be 1 or 0)\n", line_no);
                    state = S_NONE;
                    continue;
                }
                conf->pin_func_neg[PIN_16] = port;
                state = S_NONE;
            break;
            case S_PIN_17:
                if (!in_pin) {
                    in_pin = 1;
                    function = str_to_func (wrd, line_no, &lfs_flags);
                    conf->pin_func[PIN_17] = function;
                    conf->pin_lfs_flag[PIN_17] = lfs_flags;
                    continue;
                }
                in_pin = 0;
                port = strtol (wrd, &err_port, 10);
                if (((*err_port) != 0) || ((port != 0) && (port != 1))) {
                    warning ("read_config, line: %d, Invalid pin neg_value"
                            " (should be 1 or 0)\n", line_no);
                    state = S_NONE;
                    continue;
                }
                conf->pin_func_neg[PIN_17] = port;
                state = S_NONE;
            break;
            case S_EXT_OFF:
                port = strtol (wrd, &err_port, 10);
                if ((*err_port) != 0) {
                    warning ("read_config, line: %d, Invalid Ext_pins_off number\n",
                             line_no);
                    state = S_NONE;
                    continue;
                }
                conf->ext_off = port;
                state = S_NONE;
            break;
            case S_NONE:
               if (strcasecmp (wrd, "IP") == 0) {
                   state = S_IP;
                   continue;
               }
               if (strcasecmp (wrd, "Port") == 0) {
                   state = S_PORT;
                   continue;
               }
               if (strcasecmp (wrd, "LPT") == 0) {
                   state = S_LPT;
                   continue;
               }
               if (strcasecmp (wrd, "Seg_N") == 0) {
                   state = S_SEG_N;
                   continue;
               }
               if (strcasecmp (wrd, "Seg_Dot") == 0) {
                   state = S_SEG_DOT;
                   continue;
               }
               if (strcasecmp (wrd, "Debug") == 0) {
                   state = S_DEBUG;
                   continue;
               }
               if (strcasecmp (wrd, "Seg_Off") == 0) {
                   state = S_SEG_OFF;
                   continue;
               }
               if (strcasecmp (wrd, "Blink_time") == 0) {
                   state = S_BLINK;
                   continue;
               }
               if (strcasecmp (wrd, "Fuel_warning") == 0) {
                   state = S_FUEL;
                   continue;
               }
               if (strcasecmp (wrd, "PIN_1") == 0) {
                   state = S_PIN_1;
                   continue;
               }
               if (strcasecmp (wrd, "PIN_9") == 0) {
                   state = S_PIN_9;
                   continue;
               }
               if (strcasecmp (wrd, "PIN_14") == 0) {
                   state = S_PIN_14;
                   continue;
               }
               if (strcasecmp (wrd, "PIN_16") == 0) {
                   state = S_PIN_16;
                   continue;
               }
               if (strcasecmp (wrd, "PIN_17") == 0) {
                   state = S_PIN_17;
                   continue;
               }
               if (strcasecmp (wrd, "Ext_pins_off") == 0) {
                   state = S_EXT_OFF;
                   continue;
               }
               warning ("read_config, line: %d, Unknown token \"%s\"\n",
                             line_no, wrd);
            break;
            default:
                error ("read_config, undefined state!");
                free (file_buf);
                return (0);
            }
        }
    }
    free (file_buf);
    return (1);
}
